/*
 * $Id: maf.c,v 1.3 2004/11/06 18:20:09 mark Exp $
 *  
 * NAME
 *     maf - Multi-word Anagram Finder
 *
 * SYNOPSIS
 *     maf [-d file] [-f file] [-s] [-x] source_string
 *
 * DESCRIPTION
 *     The maf program finds all the words that can be formed
 *     by using the characters in the source_string argument.  When a set
 *     of words which uses all the characters in the source string 
 *     is found, the set is printed on stdout.
 *
 *     maf uses the anagram dictionary, and therefore assumes a
 *     signature prefix on each word returned from the dictionary.
 *
 *     maf accepts the following arguments:
 *        -d  use file named as a BTree dictionary file
 *        -f  use file named as a UNIX dictionary file (the default is
 *            the file words in the current directory)
 *        -s  find single-word anagrams only; default it to find multi-word
 *            and single word anagrams
 *        -x  print debug information
 *
 * NOTES
 *     Based on the description of such a program in "The Armchair Universe"
 *     by A. K. Dewdney.
 *
 * MODIFICATION HISTORY
 * Mnemonic    Date    Rel Who
 * MAF         01Oct05 1.0 MPW
 *     Written
 *   
 * Copyright (C) 2001, 2004 Mark Willson.
 *  
 * This file is part of the maf program.
 *
 * The maf program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The maf program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "strutil.h"
#include "stack.h"
#include "dictint.h"

#define TRUE 1
#define FALSE 0

#define DICTSIZE    20
#define STRMAX      80
#define KEYMAX      32
#define NAMEMAX     10

#define E_OPEN      51
#define E_POSITION  52
#define E_NEXT      53

int main(int argc, char *argv[])
{
    char dict[DICTSIZE+1],  // name of dictionary file
        token[STRMAX+1],    // holds source string
        btname[NAMEMAX+1],  // used for bt error reporting
        buff[STRMAX+1],     // string buffer for temp results
        word[KEYMAX+1],     // current word being processed
        *s;                 // used for processing command line args
    char *progname;
    
    int debug = FALSE,
        single_word = FALSE,
        err, i, found, value,
        dictype,
        pos,
        maf_error,
        npass,
        ioerr;
    int morewords,
        wordfound;
    
    char *wordstart;

    // Set default name for dictionary file
    strcpy(dict,"words");
    dictype = UNIXDICT;

    // Read command line arguments
    progname = argv[0];
    while (--argc > 0 && (*++argv)[0] == '-') {

        for (s=argv[0]+1;*s != '\0'; s++) {
            switch (*s) {
                case 'd':
                    strcpy(dict,*++argv);
                    dictype = BTREE;
                    --argc;
                    break;
                case 'f':
                    strcpy(dict,*++argv);
                    dictype = UNIXDICT;
                    --argc;
                    break;
                case 'x':
                    debug = TRUE;
                    break;
                case 's':
                    single_word = TRUE;
                    break;
                default:
                    fprintf(stderr,"%s: illegal option \"%c\"\n",progname,*s);
                    exit(-1);
            }
        }
    }
    /* assume non-switch argument is word */
    if (argc <= 0) {
        exit(0); /* nothing to do */
    }

    //Get source string
    strcpy(token,*argv);
    strlwr(token);

    // Set dictionary type
    if (!df_set_type(dictype)) {
        fprintf(stderr,"%s: bad dictionary type specified\n",progname);
        return(-1);
    }
    
    maf_error = 0;
    // Open dictionary
    if (df_open(dict) != 0) {
        maf_error = E_OPEN;
        goto trap;
    }

    //Sort characters into reverse frequency order
    strsortf(token,buff);
    strcpy(token,buff);
    if (debug) {
        fprintf(stderr,"reverse frequency sorted: %s\n",token);
    }

    //Push sentinals onto string stack
    strpush("");
    strpush("");
    ipush(0);
    
    //Set word to null
    strcpy(word,"");
    pos = 0;

    npass = 0;
    
    //While string stack is not empty
    while (strdepth() != 0) {
        //Position dictionary to word
        if (debug) {
            fprintf(stderr,"Positioning to: '%s' or position %d\n",word, pos);
        }   
        // Position dictionary
        if (df_position(word,pos) != 0) {
            maf_error = E_POSITION;
            goto trap;
        }

        //While more words in the dictionary
        if (df_next(word,&pos,&morewords) != 0) { 
            maf_error = E_NEXT;
            goto trap;
        }
        while (morewords) {
            wordfound = FALSE;
            //Does dict word contain first char of source string?
            if (strchr(word,token[0]) != NULL) {
                //check if word can be made from source string characters
                if (strcont(token,word)) {
                    if (debug) {
                        fprintf(stderr,"found word: %s\n",word);
                    }
                    // if only looking for a single word, exit search
                    // now if all characters not used
					if (!(single_word && (strlen(word) < strlen(token)))) {
                        //Push word on stack
                        strpush(word);
                        //Push source string on stack
                        strpush(token);
                        ipush(pos);
                        wordfound = TRUE;
                        //Remove used characters from source string
                        strrem(token,word,buff);
                        strcpy(token,buff);
                        //Set word to null
                        strcpy(word,"");
                        // If UNIX dict, need to start a new pass of
                        // dictionary file
                        pos = 0;
                    
                        //if source string is empty then
                        if (strlen(token) == 0) {   
                            //print word stack
                            strstkprint();
                            //pull source string
                            strcpy(token,strpull());
                            //pull word
                            strcpy(word,strpull());
                            pos = ipull();
                        }
                        else {
                            break;
                        } // if
                    } // if
                }  // if
            } // if
         
            if (df_next(word,&pos,&morewords) != 0) { 
                maf_error = E_NEXT;
                goto trap;
            }
        } // while
        if (debug) {
            fprintf(stderr,"Pass %d of dictionary complete\n",npass++);
        }
        if (!wordfound) {   
            //Pull source string
            strcpy(token,strpull());
            //Pull word
            strcpy(word,strpull());
            pos = ipull();
        } // if
    } // while
    df_close();
    exit(0);
  trap:
    if (dictype == BTREE) {
#ifdef USE_BTREE
        btcerr(&err,&ioerr,btname,buff);
        fprintf(stderr,"%s: btree error (%s) [%d] %s\n",progname,btname,err,buff);
#endif        
    }
    else {
        fprintf(stderr,"%s: dictionary file error [%d]\n",progname,maf_error);
    }
    exit(-1);
}
